module net.BurtonRadons.dig.common.treeBox;

private import net.BurtonRadons.dig.common.listBox;

/** A tree display.
  */
class TreeBox : ListBox
{
    /** A row in the tree box. */
    class Row : ListBox.Row
    {
        /** Return the parent folder for this row or null if there is none. */
        Folder parent ()
        {
            return digCommonParent;
        }
        
        /** Sort inside the parent if possible. */
        override void sortChange ()
        {
            if (parent ())
                parent ().sort ();
            else
                control.sort ();
        }
        
        /** Return whether this row is potentially visible in the tree box. */
        bit visible ()
        {
            for (Folder folder = parent (); folder; folder = folder.parent ())
                if (!folder.open ())
                    return false;
            return true;
        }
        
        /** Return the number of steps needed to get to the front of the tree. */
        int depth ()
        {
            int value = 0;
            
            for (Folder folder = parent (); folder; folder = folder.parent ())
                value ++;
                
            return value;
        }
        
        override int opCmp (Object other)
        {
            if (!(Row) other)
                return super.cmp (other);
            
            /*if (depth () != ((Row) other).depth ())
                return depth () - ((Row) other).depth ();*/
            if (cast (Folder) this)
            {
                if (cast (Folder) other)
                    return super.cmp (other);
                return -1;
            }
            else if (cast (Folder) other)
                return 1;
            return super.cmp (other);
        }
        
        override void remove ()
        {
            if (parent ())
                parent ().rowRemove (this);
            else
                control.remove (this);
        }
    
    /+
    #ifdef DoxygenMustSkipThis
    +/
    
        Folder digCommonParent; /**< Parent folder from this one or null for none. */
        
    /+
    #endif
    +/
    }
    
    /** A folder in the tree box. */
    class Folder : Row
    {
        override char [] toString ()
        {
            return this.classinfo.name;
        }
    
        /** Return whether the folder is open. */
        bit open ()
        {
            return digCommonOpen;
        }
        
        /** Assign whether the folder is open.  If the value is changed, this
          * requires redrawing the folder and everything after it.
          */
          
        void open (bit value)
        {
            bit last = open ();
            
            digCommonOpen = value;
            if (open () == last)
                return;
    
            if (open ())
                insertChildren ();
            else
                removeChildren ();        
            control.paintFromRow (this);
        }
        
        /** Return the number of child rows in this folder. */
        int rowCount ()
        {
            return digCommonRows.length;
        }
        
        /** Return the indexed child row. */
        Row row (int index)
        {
            return digCommonRows [index];
        }
        
        /** Add a row to the folder. */
        void add (Row row)
        {
            digCommonRows ~= row;
            row.digCommonParent = this;
            row.control = control;
            sort ();
        }
        
        /** Sort the contents of the row. */
        void sort ()
        {
            digCommonRows.sort;
            if (visible () && open ())
            {
                removeChildren ();
                insertChildren ();
            }
            control.paintFromRow (index () + 1);
        }
        
        /** Create a text row with a single column and add it to the folder. */
        void addText (char [] a)
        {
            char [] [1] list;
            
            list [0] = a;
            opAdd (new TextRow (list.dup));
        }
        
        /** Create a text folder with a single column and add it to the folder. */
        TextFolder addTextFolder (char [] a)
        {
            char [] [1] list;
            TextFolder item;
            
            list [0] = a;
            opAdd (item = new TextFolder (list.dup));
            return item;
        }
        
        /** Remove all children for the row from the linear list. */
        void removeChildren ()
        {
            Row [] rows = digCommonRows;
            
            control.removeFilter (delegate bit (ListBox.Row row)
            {
                for (int c; c < rows.length; c ++)
                    if (rows [c] === row)
                        return false;
                    
                return true;
            });
            
            for (int c; c < rows.length; c ++)
                if ((Folder) rows [c])
                    ((Folder) rows [c]).removeChildren ();
        }
        
        /** Add all children for the row to the linear list. */
        void insertChildren ()
        {
            Row [] rows = digCommonRows;
            int index = this.index ();
            
            for (int c = rows.length - 1; c >= 0; c --)
            {
                Row row = rows [c];
                
                control.insertAfter (row, index);
                if ((Folder) row && ((Folder) row).open ())
                    ((Folder) row).insertChildren ();
            }
        }
        
        void rowRemove (Row row)
        {
            rowFilter (delegate bit (Row compare) { return row !== compare; });
        }
        
        void rowFilter (bit delegate (Row row) func)
        {
            Row [] rows = digCommonRows;
            bit visible = this.visible () && this.open ();
            
            for (int c, d; ; c ++)
                if (c >= rows.length)
                {
                    digCommonRows = rows [0 .. d];
                    if (d != rows.length)
                        control.paintFromRow (this);
                    return;
                }
                else if (func (digCommonRows [c]))
                    rows [d ++] = rows [c];
                else if (visible)
                    control.remove (rows [c]);
        }
        
    /+
    #ifdef DoxygenMustSkipThis
    +/
    
        bit digCommonOpen; /**< Whether the folder is open. */
        Row [] digCommonRows; /**< Child rows. */
        
    /+
    #endif
    +/
    }
    
    /** A text row for the TreeBox control. */
    class TextRow : Row
    {
        char [] [] list; /**< Column values. */

        /** Assign the list, non-copying. */
        this (char [] [] list)
        {
            this.list = list;
        }
        
        /** Return a column's text value. */
        char [] columnText (Column column)
        {
            int index = column.columnIndex ();
            
            if (index >= list.length)
                return null;
            return list [index];
        }

        override int compare (ListBox.Row other, Column column)
        {
            if (cast (TextRow) other)
                return std.string.cmp (columnText (column), (cast (TextRow) other).columnText (column));
            return 0;
        }

        override int height ()
        {
            return control.textHeight ();
        }
        
        override void displayColumn (Column column, int x, int y, int width)
        {
            control.textPrintEllipses (x, y, width, 0, columnText (column));
        }
        
        override void display (int x, int y, bit selected, bit focus, bit enabled)
        {
            displayFocus (x, y, x + control.visualWidth (), y + height (), selected, focus, enabled);
            
            if (selected)
                control.textColor (Color.White);
            else
                control.textColor (Color.Black);
                
            for (int c; c < control.orderedColumnCount (); c ++)
            {
                Column column = control.orderedColumn (c);
                int ex = imin (control.visualWidth (), x + column.width);
                
                if (c == 0)
                    x += depth () * (control.treeBoxCrossWidth (false) + 3);
                displayColumn (column, x, y, ex - x);
                x = ex;
            }
        }
    }
    
    /** A folder with text label columns. */
    class TextFolder : Folder
    {
        char [] [] list; /**< Column values. */

        /** Assign the list, non-copying. */
        this (char [] [] list)
        {
            this.list = list;
        }
        
        /** Return a column's text value. */
        char [] columnText (Column column)
        {
            int index = column.columnIndex ();
            
            if (index >= list.length)
                return null;
            return list [index];
        }

        override int compare (ListBox.Row other, Column column)
        {
            if (cast (TextFolder) other)
                return std.string.cmp (columnText (column), (cast (TextFolder) other).columnText (column));
            return 0;
        }

        override int height ()
        {
            return control.textHeight ();
        }
        
        override void displayColumn (Column column, int x, int y, int width)
        {
            if (column.orderedIndex () == 0)
            {
                int cw, ch;
                
                control.treeBoxCrossSize (cw, ch, open ());
                int c = depth () * (cw + 3);
                x += c;
                width -= c;
                control.treeBoxCrossDraw (x , y + (height () - ch) / 2, open ());
                x += cw + 3;
            }
            
            control.textPrintEllipses (x, y, width, 0, columnText (column));
        }
    }    
    
    /** Assign default parameters and create a single sorted column, "Path",
      * then hide the column header.
      */
      
    this (Control parent)
    {
        super (parent);
        columnHeaderShown (false);
        with (addColumn ("Path", 100))
            sort ();
        onLButtonDown.add (&digCommonDoCrossTouch);
        noSelection (false);
    }
    
    override void funcFocusSelectExact ()
    {
        Folder folder = (Folder) rowFocus ();
        
        if (folder)
            folder.open (!folder.open ());
    }
    
    /** Create a text folder and add it to the root. */
    TextFolder addTextFolder (char [] a)
    {
        char [] [1] list;
        TextFolder item;
        
        list [0] = a;
        opAdd (item = new TextFolder (list.dup));
        return item;
    }
    
/+
#ifdef DoxygenMustSkipThis
+/

    void digCommonDoCrossTouch (Event event)
    {
        Folder folder = (Folder) findRow (event.y);
        
        if (!folder)
            return;
        
        int crossWidth = treeBoxCrossWidth (folder.open ());
        
        event.x -= folder.depth () * (crossWidth + 3);
        
        if (event.x >= 0 && event.x < crossWidth + 6)
            folder.open (!folder.open ());
    }

/+
#endif
+/
}
